<?php
/* Copyright (C) 2005	  	Rodolphe Quiedeville <rodolphe@quiedeville.org>
 * Copyright (C) 2005-2012 	Laurent Destailleur	<eldy@users.sourceforge.net>
 * Copyright (C) 2005-2012 	Regis Houssin		<regis.houssin@capnetworks.com>
 * Copyright (C) 2018-2021 	Charlene Benke		<charlie@patas-monkey.com>
 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 * or see http://www.gnu.org/
 */

/**
 *	\file	   myschedule/core/modules/user/doc/pdf_user_bulle.modules.php
 *	\ingroup	myschedule
 *	\brief	  Fichier de la classe permettant de generer pdf de contact martchr
 */
require_once DOL_DOCUMENT_ROOT.'/core/modules/user/modules_user.class.php';
require_once DOL_DOCUMENT_ROOT.'/core/lib/company.lib.php';
require_once DOL_DOCUMENT_ROOT.'/core/lib/pdf.lib.php';
require_once DOL_DOCUMENT_ROOT.'/core/lib/date.lib.php';

dol_include_once('/myschedule/class/dailytime.class.php');

/**
 *	Classe permettant de generer les borderaux envoi au modele Rouget
 */
class pdf_user_bulle extends ModelePDFUser
{
	var $emetteur;	// Objet societe qui emet


	/**
	 *	Constructor
	 *
	 *	@param	DoliDB	$db		Database handler
	 */
	function __construct($db=0)
	{
		global $conf, $langs, $mysoc;

		$langs->load("myschedule@myschedule");

		$this->db = $db;
		$this->name = "Bulle";
		$this->description = $langs->trans("DocumentModelUserBulleMySchedule");

		$this->type = 'pdf';
		$formatarray=pdf_getFormat();
		$this->page_largeur = $formatarray['width'];
		$this->page_hauteur = $formatarray['height'];
		$this->format = array($this->page_largeur, $this->page_hauteur);
		$this->marge_gauche=isset($conf->global->MAIN_PDF_MARGIN_LEFT)?$conf->global->MAIN_PDF_MARGIN_LEFT:10;
		$this->marge_droite=isset($conf->global->MAIN_PDF_MARGIN_RIGHT)?$conf->global->MAIN_PDF_MARGIN_RIGHT:10;
		$this->marge_haute =isset($conf->global->MAIN_PDF_MARGIN_TOP)?$conf->global->MAIN_PDF_MARGIN_TOP:10;
		$this->marge_basse =isset($conf->global->MAIN_PDF_MARGIN_BOTTOM)?$conf->global->MAIN_PDF_MARGIN_BOTTOM:10;

		$this->option_logo = 1;

		// Recupere emmetteur
		$this->emetteur=$mysoc;
		if (! $this->emetteur->pays_code) 
			$this->emetteur->pays_code=substr($langs->defaultlang,-2);	// By default if not defined

		// Defini position des colonnes
		$this->posxdesc=$this->marge_gauche+1;
		$this->posxqtyordered=$this->page_largeur - $this->marge_droite - 70;
		$this->posxqtytoship=$this->page_largeur - $this->marge_droite - 35;
	}

	/**
	 *	Function to build pdf onto disk
	 *
	 *	@param		Object		&$object			Object expedition to generate (or id if old method)
	 *	@param		Translate	$outputlangs		Lang output object
	 *  @param		string		$srctemplatepath	Full path of source filename for generator using a template file
	 *  @param		int			$hidedetails		Do not show line details
	 *  @param		int			$hidedesc			Do not show desc
	 *  @param		int			$hideref			Do not show ref
	 *  @param		object		$hookmanager		Hookmanager object
	 *  @return	 int		 					1=OK, 0=KO
	 */
	function write_file(&$object, $outputlangs, $srctemplatepath='', $hidedetails=0, $hidedesc=0, $hideref=0, $hookmanager=false)
	{
		global $user, $conf, $langs;


		if (! is_object($outputlangs)) $outputlangs=$langs;
		// For backward compatibility with FPDF, force output charset to ISO, because FPDF expect text to be encoded in ISO
		if (! empty($conf->global->MAIN_USE_FPDF)) $outputlangs->charset_output='ISO-8859-1';

		$outputlangs->load("main");
		$outputlangs->load("dict");
		$outputlangs->load("companies");
		$outputlangs->load("contact");
		$outputlangs->load("myschedule@myschedule");

		if ($conf->user->dir_output) {
			// Definition de $dir et $file
			if ($object->specimen) {
				$dir = $conf->user->dir_output;
				$file = $dir . "/SPECIMEN.pdf";
			} else {
				$expref = dol_sanitizeFileName($object->ref);
				$datedeb = dol_mktime(
								0, 0, 0,
								GETPOST('datedebmonth', 'int'),
								GETPOST('datedebday', 'int'),
								GETPOST('datedebyear', 'int')
				);
				$datefin = dol_mktime(
								0, 0, 0,
								GETPOST('datefinmonth', 'int'),
								GETPOST('datefinday', 'int'),
								GETPOST('datefinyear', 'int')
				);	

				$namedatedeb=dol_print_date(
								$datedeb, "%d-%m-%y", false, $outputlangs, true
				);
				$namedatefin=dol_print_date(
								$datefin, "%d-%m-%y", false, $outputlangs, true
				);

				$dir = $conf->user->dir_output."/".$expref;
				switch($conf->global->MYSCHEDULE_FormatDisplayMode) {
					case 'all':
						$formatdisplaymode = "DT";
						break;
					case 'allwithouthour':
						$formatdisplaymode = "D";
						break;
					case 'allhourmin':
						$formatdisplaymode = "T";
						break;

				}
				$file = $dir. "/" .$expref.".".$formatdisplaymode."(".$namedatedeb."_".$namedatefin.").pdf";
			}


			if (! file_exists($dir)) {
				if (dol_mkdir($dir) < 0) {
					$this->error=$langs->transnoentities("ErrorCanNotCreateDir", $dir);
					return 0;
				}
			}

			if (file_exists($dir)) {
				$nblignes = count($object->lines);

				$pdf=pdf_getInstance($this->format);
				$default_font_size = pdf_getPDFFontSize($outputlangs);
				$heightforinfotot = 0;	// Height reserved to output the info and total part
				// Height reserved to output the free text on last page
				$heightforfreetext= (isset($conf->global->MAIN_PDF_FREETEXT_HEIGHT)?$conf->global->MAIN_PDF_FREETEXT_HEIGHT:5);	
				$heightforfooter = $this->marge_basse + 8;	// Height reserved to output the footer (value include bottom margin)
				$pdf->SetAutoPageBreak(1, 0);

				if (class_exists('TCPDF')) {
					$pdf->setPrintHeader(false);
					$pdf->setPrintFooter(false);
				}
				$pdf->SetFont(pdf_getPDFFont($outputlangs));
				// Set path to the background PDF File
				if (empty($conf->global->MAIN_DISABLE_FPDI) && ! empty($conf->global->MAIN_ADD_PDF_BACKGROUND)) {
					$pagecount = $pdf->setSourceFile($conf->mycompany->dir_output.'/'.$conf->global->MAIN_ADD_PDF_BACKGROUND);
					$tplidx = $pdf->importPage(1);
				}

				$pdf->Open();
				$pagenb=0;
				$pdf->SetDrawColor(128, 128, 128);

				//$pdf->AliasNbPages();

				$pdf->SetTitle($outputlangs->convToOutputCharset($expref.".".$precmonth));
				$pdf->SetSubject($outputlangs->transnoentities("UserCardReport"));
				$pdf->SetCreator("Dolibarr ".DOL_VERSION);
				$pdf->SetAuthor($outputlangs->convToOutputCharset($user->getFullName($outputlangs)));
				$pdf->SetKeyWords($outputlangs->convToOutputCharset($object->ref)." ".$outputlangs->transnoentities("Sending"));
				if (! empty($conf->global->MAIN_DISABLE_PDF_COMPRESSION)) $pdf->SetCompression(false);

				$pdf->SetMargins($this->marge_gauche, $this->marge_haute, $this->marge_droite);   // Left, Top, Right

				// New page
				$pdf->AddPage();
				if (! empty($tplidx)) $pdf->useTemplate($tplidx);
				$pagenb++;
				$this->_pagehead($pdf, $object, 1, $outputlangs, $hookmanager);
				$pdf->SetFont('', '', $default_font_size - 1);
				$pdf->MultiCell(0, 3, '');		// Set interline to 3
				$pdf->SetTextColor(0, 0, 0);

				$tab_top = 90;
				$tab_top_newpage = (empty($conf->global->MAIN_PDF_DONOTREPEAT_HEAD)?42:10);
				$tab_height = 130;
				$tab_height_newpage = 150;

				if (! empty($object->note_public) || ! empty($object->tracking_number)) {
					$tab_top = 88;

					// Affiche notes
					if (! empty($object->note_public)) {
						$pdf->SetFont('', '', $default_font_size - 1);   // Dans boucle pour gerer multi-page
						$pdf->writeHTMLCell(190, 3, $this->posxdesc-1, $tab_top, dol_htmlentitiesbr($object->note_public), 0, 1);


						$nexY = $pdf->GetY();
						$height_note=$nexY-$tab_top;

						// Rect prend une longueur en 3eme param
						$pdf->SetDrawColor(192, 192, 192);
						$pdf->Rect(
										$this->marge_gauche, $tab_top-1, 
										$this->page_largeur-$this->marge_gauche-$this->marge_droite, 
										$height_note+1
						);

						$tab_height = $tab_height - $height_note;
						$tab_top = $nexY+6;

					}
				} else
					$height_note=0;

				$curY = $tab_top + 7;

				// récup des heures hedbo habituel de la personne
				$formatdisplaymode = $conf->global->MYSCHEDULE_FormatDisplayMode;
				$daily = new Dailytime($this->db);
				$dailyUser = $daily->fetchArray($object->id);


				$pdf->SetXY($this->posxdesc, $curY);
				$pdf->MultiCell(
								($this->posxqtyordered - $this->posxdesc), 3, 
								$outputlangs->transnoentities("TimeToMade"), '', 'L'
				);
				

				$jourTraite = $datedeb;
				$totalLine=0;
				while ($jourTraite < $datefin) {
					$totalLine+=$dailyUser[date('w', $jourTraite)];
					$jourTraite = strtotime(date('d-m-Y', strtotime("+1 day", $jourTraite)));
				}
				$pdf->SetXY($this->posxqtytoship, $curY);
				$pdf->MultiCell(
								($this->page_largeur - $this->marge_droite - $this->posxqtytoship), 3,
				 				convertSecondToTime($totalLine, $formatdisplaymode, ($formatdisplaymode=="allhourmin"?86400:28800)), 
				 				'', 'R'
				 );


				$curY+=5; 
				$pdf->SetXY($this->posxdesc, $curY);
				$pdf->MultiCell(
								($this->posxqtyordered - $this->posxdesc), 3, 
								$outputlangs->transnoentities("TimePlanned"), '', 'L'
				);
				$totalplanned=0;
				if ($conf->global->MYSCHEDULE_DisplayFichinterPlanning) {
					// la requete pour r�cup�rer les valeurs
					$sql = "SELECT sum(duration_planned) as total";
					$sql.= " FROM ".MAIN_DB_PREFIX."fichinterdet as fd";
					$sql.= " ,".MAIN_DB_PREFIX."fichinterdet_schedule as fds";
					$sql.= " WHERE fds.fk_user = ".$object->id;
					$sql.= " AND fds.fk_fichinterdet = fd.rowid";

					$sql.= " AND fd.date BETWEEN '".$this->db->idate($datedeb)."'";
					$sql.= "  AND '".$this->db->idate($datefin)."'";
					
					//$pdf->SetXY(50, $curY);
					//$pdf->MultiCell((100), 3,$sql, '', 'R');


					$result = $this->db->query($sql);
					if ($result) {
						$obj = $this->db->fetch_object($result);
						$totalplanned+=$obj->total;
					}
				}

				if ($conf->global->MYSCHEDULE_DisplayProjectPlanning) {
					// la requete pour r�cup�rer les valeurs
					$sql = "SELECT count(rowid) as nbelem, count(fk_task) as nbtask, sum(task_duration) as total";
					$sql.= " FROM ".MAIN_DB_PREFIX."projet_task_schedule as ptt";
					$sql.= " WHERE fk_user = ".$object->id;
					$sql.= " AND task_date BETWEEN '".$this->db->idate($datedeb)."'";
					$sql.= "  AND '".$this->db->idate($datefin)."'";

					$result = $this->db->query($sql);
					if ($result) {
						$obj = $this->db->fetch_object($result);
						$totalplanned+=$obj->total;
					}
				}

				$pdf->SetXY($this->posxqtytoship, $curY);
				$pdf->MultiCell(
								($this->page_largeur - $this->marge_droite - $this->posxqtytoship), 3,
								convertSecondToTime($totalplanned, $formatdisplaymode, ($formatdisplaymode=="allhourmin"?86400:28800)), 
								'', 'R'
				);

				$curY+=5; 
				$pdf->SetXY($this->posxdesc, $curY);
				$pdf->MultiCell(
								($this->posxqtyordered - $this->posxdesc), 3, 
								$outputlangs->transnoentities("TimeMade"), '', 'L'
				);

				$totalmade=0;
				if ($conf->global->MYSCHEDULE_DisplayFichinterPlanning) {
					// la requete pour r�cup�rer les valeurs
					$sql = "SELECT sum(duration_made) as total";
					$sql.= " FROM ".MAIN_DB_PREFIX."fichinterdet as fd";
					$sql.= " ,".MAIN_DB_PREFIX."fichinterdet_schedule as fds";
					$sql.= " WHERE fds.fk_user = ".$object->id;
					$sql.= " AND fds.fk_fichinterdet = fd.rowid";

					$sql.= " AND fd.date BETWEEN '".$this->db->idate($datedeb)."'";
					$sql.= "  AND '".$this->db->idate($datefin)."'";
					
					//$pdf->SetXY(50, $curY);
					//$pdf->MultiCell((100), 3,$sql, '', 'R');


					$result = $this->db->query($sql);
					if ($result) {
						$obj = $this->db->fetch_object($result);
						$totalmade+=$obj->total;
					}
				}

				if ($conf->global->MYSCHEDULE_DisplayProjectPlanning) {
					// la requete pour r�cup�rer les valeurs
					$sql = "SELECT count(rowid) as nbelem, count(fk_task) as nbtask, sum(task_duration) as total";
					$sql.= " FROM ".MAIN_DB_PREFIX."projet_task_time as ptt";
					$sql.= " WHERE fk_user = ".$object->id;
					$sql.= " AND task_date BETWEEN '".$this->db->idate($datedeb)."'";
					$sql.= "  AND '".$this->db->idate($datefin)."'";

					$result = $this->db->query($sql);
					if ($result) {
						$obj = $this->db->fetch_object($result);
						$totalmade+=$obj->total;
					}
				}

				$pdf->SetXY($this->posxqtytoship, $curY);
				$pdf->MultiCell(
								($this->page_largeur - $this->marge_droite - $this->posxqtytoship), 3,
				 				convertSecondToTime($totalmade, $formatdisplaymode, ($formatdisplaymode=="allhourmin"?86400:28800)), 
				 				'', 'R'
				 );

				$curY+=5; 
				$pdf->SetXY($this->posxdesc, $curY);
				$pdf->MultiCell(
								($this->posxqtyordered - $this->posxdesc), 3, 
								$outputlangs->transnoentities("MissingTime"), '', 'L'
				);

				// la requete pour r�cup�rer les valeurs
				$sql = "SELECT task_date, date_format( task_date, '%w') as numjour, sum(task_duration) as total";
				$sql.= " FROM ".MAIN_DB_PREFIX."projet_task_time as ptt";
				$sql.= " WHERE fk_user = ".$object->id;
				$sql.= " AND task_date BETWEEN '".$this->db->idate($datedeb)."'";
				$sql.= "  AND '".$this->db->idate($datefin)."'";
				$sql.= " GROUP BY task_date, date_format( task_date, '%w')";

				$result = $this->db->query($sql);
				if ($result) {
					$i = 0;
					$num = $this->db->num_rows($result);

					$missingtimeperiod=0;
					while ($i < $num) {
						$obj = $this->db->fetch_object($result);
		//				print $obj->task_date."- ".$obj->numjour."- ".$obj->total."- ".$dailyUser[$obj->numjour]."<br>";
						if ($dailyUser[$obj->numjour] - $obj->total > 0)
							$missingtimeperiod+= ($dailyUser[$obj->numjour] - $obj->total);
						$i++;
					}
					$pdf->SetXY($this->posxqtytoship, $curY);
					$pdf->MultiCell(
									($this->page_largeur - $this->marge_droite - $this->posxqtytoship), 3,
					 				convertSecondToTime($missingtimeperiod, $formatdisplaymode, ($formatdisplaymode=="allhourmin"?86400:28800)), 
					 				'', 'R'
					 );
				}


				$curY+=5; 
				$pdf->SetXY($this->posxdesc, $curY);
				$pdf->MultiCell(
								($this->posxqtyordered - $this->posxdesc), 3, 
								$outputlangs->transnoentities("OverTime"), '', 'L'
				);

				// la requete pour r�cup�rer les valeurs
				$sql = "SELECT task_date, date_format( task_date, '%w') as numjour, sum(task_duration) as total";
				$sql.= " FROM ".MAIN_DB_PREFIX."projet_task_time as ptt";
				$sql.= " WHERE fk_user = ".$object->id;
				$sql.= " AND task_date BETWEEN '".$this->db->idate($datedeb)."'";
				$sql.= "  AND '".$this->db->idate($datefin)."'";
				$sql.= " GROUP BY task_date, date_format( task_date, '%w')";

				$result = $this->db->query($sql);
				if ($result) {
					$i = 0;
					$num = $this->db->num_rows($result);

					$overtimeperiod=0;
					while ($i < $num) {
						$obj = $this->db->fetch_object($result);
		//				print $obj->task_date."- ".$obj->numjour."- ".$obj->total."- ".$dailyUser[$obj->numjour]."<br>";
						if ($obj->total - $dailyUser[$obj->numjour] > 0)
							$overtimeperiod+= ($obj->total - $dailyUser[$obj->numjour]);
						$i++;
					}
					$pdf->SetXY($this->posxqtytoship, $curY);
					$pdf->MultiCell(
									($this->page_largeur - $this->marge_droite - $this->posxqtytoship), 3,
					 				convertSecondToTime($overtimeperiod, $formatdisplaymode, ($formatdisplaymode=="allhourmin"?86400:28800)), 
					 				'', 'R'
					 );
				}


				// affichage des cong�s regroup� par type (table llx_holliday et llx_c_holliday_types )
				$sql="SELECT distinct cht.rowid, cht.label ";
				$sql.= " FROM ".MAIN_DB_PREFIX."c_holiday_types as cht, ".MAIN_DB_PREFIX."holiday as h";
				$sql.= " WHERE h.fk_user=".$object->id;
				// sur la p�riode consid�r�e (datedeb, datefin)
				$sql.= " AND ( h.date_debut between '". $this->db->idate($datedeb)."' and '".$this->db->idate($datefin)."'";
				$sql.= " OR h.date_fin between '". $this->db->idate($datedeb)."' and '".$this->db->idate($datefin)."')";
				$sql.= " AND h.fk_type = cht.rowid";



				$resql = $this->db->query($sql);
				if ($resql) { 
					$num = $this->db->num_rows($resql);
					$i=0;
					while ($i < $num) {
						$obj = $this->db->fetch_object($resql);
						$curY+=5; 
						$pdf->SetXY($this->posxdesc, $curY);
						$pdf->MultiCell(
										($this->posxqtyordered - $this->posxdesc), 3, 
										$obj->label, '', 'L'
						);
						$totalconge = 0;
						// Liste des cong�s sur la p�riode (table llx_holliday)
						$sql="SELECT h.date_debut, h.date_fin, halfday ";
						$sql.= " FROM ".MAIN_DB_PREFIX."holiday as h";
						// uniquement les types de cong�s du collaborateur
						$sql.= " WHERE h.fk_user=".$object->id;
						$sql.= " AND h.fk_type = ".$obj->rowid;
						// sur la p�riode consid�r�e (datedeb, datefin)
						$sql.= " AND ( h.date_debut between '". $this->db->idate($datedeb)."' and '".$this->db->idate($datefin)."'";
						$sql.= " OR h.date_fin between '". $this->db->idate($datedeb)."' and '".$this->db->idate($datefin)."')";
						$resconge = $this->db->query($sql);

						if ($resconge) { 
							$numconge = $this->db->num_rows($resconge);
							$j=0;
							while ($j < $numconge) {
								$objconges = $this->db->fetch_object($resconge);
								$jourTraite = $objconges->date_debut;

								while ($this->db->idate($jourTraite) <= $this->db->idate($objconges->date_fin)) {
									$datetraite = new DateTime($jourTraite);

									// print $jourTraite.">>"." ".date('w', strtotime($jourTraite))." ".$dailyUser[date('w', strtotime($jourTraite))].'<br>';
									$totalconge+=$dailyUser[date('w', strtotime($jourTraite))];
									
									// on passe au jour suivant
									$datetraite->add(new DateInterval('P1D'));
									$jourTraite = $datetraite->format('Y-m-d');
								}
								$j++;
							}
							$pdf->SetXY($this->posxqtytoship, $curY);
							$pdf->MultiCell(
											($this->page_largeur - $this->marge_droite - $this->posxqtytoship), 3,
							 				convertSecondToTime($totalconge, $formatdisplaymode, ($formatdisplaymode=="allhourmin"?86400:28800)), 
							 				'', 'R'
							 );

						}
						$i++;
					}
				}

				// tableau des temps
				$this->_tableau(
								$pdf, $tab_top, 
								60, 
								0, $outputlangs, 0, 0
				);

				// on affiche le r�cap des interventions
				if ($conf->global->MYSCHEDULE_DisplayFichinterPlanning) {
					
					$curY=160;
					$sql = 'SELECT s.nom, s.address, s.zip, s.town, s.phone, f.ref, f.description as descinter, ';
					$sql.= ' ft.description as descinterdet, ft.fk_fichinter, fds.duration_planned, ft.duree, fds.duration_made, ft.date as date_intervention';
					$sql.= ' FROM '.MAIN_DB_PREFIX.'fichinterdet as ft, '.MAIN_DB_PREFIX.'fichinter as f';
					$sql.= " ,".MAIN_DB_PREFIX."fichinterdet_schedule as fds";
					$sql.= ' , '.MAIN_DB_PREFIX.'societe as s';
					$sql.= " WHERE ft.date BETWEEN '".$this->db->idate($datedeb)."' AND '".$this->db->idate($datefin)."'";
					$sql.= ' AND ft.rowid=fds.fk_fichinterdet ';
					$sql.= ' AND ft.fk_fichinter = f.rowid';
					$sql.= ' AND f.fk_soc= s.rowid';
					$sql.= ' AND fds.fk_user = '.$object->id;
					$sql.= ' ORDER BY ft.date ASC';
					//$pdf->SetXY(10, $curY);
					//$pdf->MultiCell(60, 10,$sql, '', 'L');

					$res= $this->db->query($sql);

					if ($res) { 
						$num= $this->db->num_rows($res);
						$j=0;
						while ($j < $num) {
							$obj = $this->db->fetch_object($res);
							// on trace une ligne
							$pdf->line(
											$this->marge_gauche, $curY-1+($j*17), 
											$this->page_largeur-$this->marge_droite, $curY-1+($j*17));

							$pdf->SetXY(10, $curY+($j*17));
							if ($obj->duration_planned > 0) {
								$timeinter= " (".convertSecondToTime($obj->duration_planned, "allhourmin");
								if ($obj->duration_made > 0)
									$timeinter.= " / ".convertSecondToTime($obj->duration_made, "allhourmin");
								$timeinter.= " )";
							} else
								$timeinter= $outputlangs->transnoentities("Toplan")." : ".convertSecondToTime($obj->duree, "allhourmin");

							$pdf->MultiCell(
											140, 10,
							 				$obj->date_intervention." -> ".$obj->descinterdet.$timeinter, 
							 				'', 'L'
							 );

							$pdf->SetXY(10, $curY+4+($j*17));
							$pdf->MultiCell(
											60, 10,
							 				$obj->nom."\n".$obj->address."\n".$obj->zip." ".$obj->town."\n".$obj->phone, 
							 				'', 'L'
							 );

							$pdf->SetXY(80, $curY+4+($j*17));
							$pdf->MultiCell(
											100, 10,
							 				$obj->ref." - ".$obj->descinter, 
							 				'', 'L'
							 );
							$j++;
						}
					}
				}

				$bottomlasttab=$this->page_hauteur - $heightforfooter - $heightforfreetext - $heightforinfotot + 1;

				// Pied de page
				$this->_pagefoot($pdf, $object, $outputlangs);
				//$pdf->AliasNbPages();
				
				
				// on ajoute la page des intervention si c'est demand�

				$pdf->Close();

				$pdf->Output($file, 'F');
				if (! empty($conf->global->MAIN_UMASK))
					@chmod($file, octdec($conf->global->MAIN_UMASK));

				return 1;
			} else {
				$this->error=$langs->transnoentities("ErrorCanNotCreateDir", $dir);
				return 0;
			}
		} else {
			$this->error=$langs->transnoentities("ErrorConstantNotDefined", "EXP_OUTPUTDIR");
			return 0;
		}
		$this->error=$langs->transnoentities("ErrorUnknown");
		return 0;   // Erreur par defaut
	}

	/**
	 *   Show table for lines
	 *
	 *   @param		PDF			&$pdf	 		Object PDF
	 *   @param		string		$tab_top		Top position of table
	 *   @param		string		$tab_height		Height of table (rectangle)
	 *   @param		int			$nexY			Y
	 *   @param		Translate	$outputlangs	Langs object
	 *   @param		int			$hidetop		Hide top bar of array
	 *   @param		int			$hidebottom		Hide bottom bar of array
	 *   @return	void
	 */
	function _tableau(&$pdf, $tab_top, $tab_height, $nexY, $outputlangs, $hidetop=0, $hidebottom=0)
	{
		// Force to disable hidetop and hidebottom
		$hidebottom=0;
		if ($hidetop) $hidetop=-1;

		$default_font_size = pdf_getPDFFontSize($outputlangs);

		// Amount in (at tab_top - 1)
		$pdf->SetTextColor(0, 0, 0);
		$pdf->SetFont('', '', $default_font_size - 1);

		// Output Rect
		$this->printRect($pdf, $this->marge_gauche, $tab_top, $this->page_largeur-$this->marge_gauche-$this->marge_droite, $tab_height, $hidetop, $hidebottom);	// Rect prend une longueur en 3eme param et 4eme param

		$pdf->SetDrawColor(128, 128, 128);
		$pdf->SetFont('', '', $default_font_size - 1);

		if (empty($hidetop)) {
			$pdf->line($this->marge_gauche, $tab_top+5, $this->page_largeur-$this->marge_droite, $tab_top+5);

			$pdf->SetXY($this->posxdesc-1, $tab_top+1);
			$pdf->MultiCell(
							$this->posxqtyordered - $this->posxdesc, 2, 
							$outputlangs->transnoentities("Description"), 
							'', 'L'
			);
		}

		$pdf->line($this->posxqtytoship-1, $tab_top, $this->posxqtytoship-1, $tab_top + $tab_height);
		if (empty($hidetop)) {
			$pdf->SetXY($this->posxqtytoship, $tab_top+1);
			$pdf->MultiCell(
							($this->page_largeur - $this->marge_droite - $this->posxqtytoship), 2, 
							$outputlangs->transnoentities("Value"),
							'', 'R'
			);
		}
	}

	/**
	 *  Show top header of page.
	 *
	 *  @param	PDF			&$pdf	 		Object PDF
	 *  @param  Object		$object	 	Object to show
	 *  @param  int			$showaddress	0=no, 1=yes
	 *  @param  Translate	$outputlangs	Object lang for output
	 *  @return	void
	 */
	function _pagehead(&$pdf, $object, $showaddress, $outputlangs)
	{
		global $conf, $langs; //, $mysoc;
		$default_font_size = pdf_getPDFFontSize($outputlangs);
		$outputlangs->load("myschedule@myschedule");

		pdf_pagehead($pdf, $outputlangs, $this->page_hauteur);

		//Affiche le filigrane brouillon - Print Draft Watermark
		if ($object->statut==0 && (! empty($conf->global->MYSCHEDULE_DRAFT_WATERMARK)) )
			pdf_watermark(
							$pdf, $outputlangs, $this->page_hauteur, $this->page_largeur, 
							'mm', $conf->global->MYSCHEDULE_DRAFT_WATERMARK
			);


		//Prepare la suite
		$pdf->SetTextColor(0, 0, 60);
		$pdf->SetFont('', 'B', $default_font_size + 3);

		$posx=$this->page_largeur-$this->marge_droite-100;
		$posy=$this->marge_haute;

		$pdf->SetXY($this->marge_gauche, $posy);

		// Logo
		$logo=$conf->mycompany->dir_output.'/logos/'.$this->emetteur->logo;
		if ($this->emetteur->logo) {
			if (is_readable($logo)) {
				$height=pdf_getHeightForLogo($logo);
				$pdf->Image($logo, $this->marge_gauche, $posy, 0, $height);	// width=0 (auto)
			} else {
				$pdf->SetTextColor(200, 0, 0);
				$pdf->SetFont('', 'B', $default_font_size - 2);
				$pdf->MultiCell(100, 3, $outputlangs->transnoentities("ErrorLogoFileNotFound", $logo), 0, 'L');
				$pdf->MultiCell(100, 3, $outputlangs->transnoentities("ErrorGoToGlobalSetup"), 0, 'L');
			}
		} else {
			$text=$this->emetteur->name;
			$pdf->MultiCell(100, 4, $outputlangs->convToOutputCharset($text), 0, 'L');
		}

		$posx=$this->page_largeur - 100 - $this->marge_droite;
		$posy=$this->marge_haute;

		$pdf->SetFont('', 'B', $default_font_size + 2);
		$pdf->SetXY($posx, $posy);
		$pdf->SetTextColor(0, 0, 60);
		$title=$outputlangs->transnoentities("UserScheduleReport");
		$pdf->MultiCell(100, 4, $title, '', 'R');
		$posy+=1;

		$pdf->SetFont('', '', $default_font_size + 1);

		$posy+=4;
		$pdf->SetXY($posx, $posy);
		$pdf->SetTextColor(0, 0, 60);
		$pdf->MultiCell(100, 4, $outputlangs->transnoentities("RefUser") ." : ".$object->ref, '', 'R');

		//p�riode d'impression
		$datedeb = dol_mktime(
						0, 0, 0,
						GETPOST('datedebmonth', 'int'),
						GETPOST('datedebday', 'int'),
						GETPOST('datedebyear', 'int')
		);
		$datefin = dol_mktime(
						0, 0, 0,
						GETPOST('datefinmonth', 'int'),
						GETPOST('datefinday', 'int'),
						GETPOST('datefinyear', 'int')
		);

		

		$posy+=4;
		$pdf->SetXY($posx, $posy);
		$pdf->SetTextColor(0, 0, 60);
		$pdf->MultiCell(
						100, 4, 
						$outputlangs->transnoentities("DatePeriodFrom")." : ".dol_print_date(
										$datedeb, "daytext", false, $outputlangs, true
						), '', 'R'
		);

		$posy+=4;
		$pdf->SetXY($posx, $posy);
		$pdf->SetTextColor(0, 0, 60);
		$pdf->MultiCell(
						100, 4, 
						$outputlangs->transnoentities("DatePeriodTo")." : ".dol_print_date(
										$datefin, "daytext", false, $outputlangs, true
						), '', 'R'
		);
		$yoff=25;

		if ($showaddress) {
			// Sender properties
			$carac_emetteur='';
		 	// Add internal contact of origin element if defined
			$arrayidcontact=array();
			if (! empty($origin) && is_object($object->$origin)) 
				$arrayidcontact=$object->$origin->getIdContact('internal', 'SALESREPFOLL');
		 	if (count($arrayidcontact) > 0) {
		 		$object->fetch_user($arrayidcontact[0]);
		 		$carac_emetteur .= ($carac_emetteur ? "\n" : '' ).$outputlangs->transnoentities("Name").": ";
		 		$carac_emetteur .= $outputlangs->convToOutputCharset($object->user->getFullName($outputlangs))."\n";
		 	}

		 	$carac_emetteur .= pdf_build_address($outputlangs, $this->emetteur);

			// Show sender
			$posx=$this->marge_gauche;
			$posy=42;
			$hautcadre=40;
			if (! empty($conf->global->MAIN_INVERT_SENDER_RECIPIENT)) 
				$posx=$this->page_largeur - 80 - $this->marge_droite;

			// Show sender frame
			$pdf->SetTextColor(0, 0, 0);
			$pdf->SetFont('', '', $default_font_size - 2);
			$pdf->SetXY($posx, $posy-5);
			$pdf->MultiCell(66, 5, $outputlangs->transnoentities("Sender").":", 0, 'L');
			$pdf->SetXY($posx, $posy);
			$pdf->SetFillColor(230, 230, 230);
			$pdf->MultiCell(82, $hautcadre, "", 0, 'R', 1);

			// Show sender name
			$pdf->SetXY($posx+2, $posy+3);
			$pdf->SetTextColor(0, 0, 60);
			$pdf->SetFont('', 'B', $default_font_size);
			$pdf->MultiCell(80, 3, $outputlangs->convToOutputCharset($this->emetteur->name), 0, 'L');

			// Show sender information
			$pdf->SetFont('', '', $default_font_size - 1);
			$pdf->SetXY($posx+2, $posy+8);
			$pdf->MultiCell(80, 4, $carac_emetteur, 0, 'L');

			// Recipient name
			$carac_user_name=$outputlangs->convToOutputCharset($object->firstname." ".$object->lastname);

			$carac_user=$object->address."\n";
			$carac_user.=$object->zip." ".$object->town."\n";

			// Show recipient
			$widthrecbox=100;
			if ($this->page_largeur < 210) $widthrecbox=84;	// To work with US executive format
			$posy=42;
			$posx=$this->page_largeur - $this->marge_droite - $widthrecbox;
			if (! empty($conf->global->MAIN_INVERT_SENDER_RECIPIENT)) 
				$posx=$this->marge_gauche;

			// Show recipient frame
			$pdf->SetTextColor(0, 0, 0);
			$pdf->SetFont('', '', $default_font_size - 2);
			$pdf->SetXY($posx, $posy-5);
			$pdf->MultiCell($widthrecbox, 4, $outputlangs->transnoentities("User").":", 0, 'L');
			$pdf->Rect($posx, $posy, $widthrecbox, $hautcadre);
			$pdf->SetTextColor(0, 0, 0);

			// Show recipient name
			$pdf->SetXY($posx+2, $posy+3);
			$pdf->SetFont('', 'B', $default_font_size);
			$pdf->MultiCell($widthrecbox, 4, $carac_user_name, 0, 'L');

			// Show recipient information
			$pdf->SetFont('', '', $default_font_size - 1);
			$pdf->SetXY($posx+2, $posy+4+(dol_nboflines_bis($carac_user_name, 50)*4));
			$pdf->MultiCell($widthrecbox, 4, $carac_user, 0, 'L');
		}

	}

	/**
	 *   	Show footer of page. Need this->emetteur object
	 *
	 *   	@param	PDF			&$pdf	 			PDF
	 * 		@param	Object		$object				Object to show
	 *	  @param	Translate	$outputlangs		Object lang for output
	 *	  @param	int			$hidefreetext		1=Hide free text
	 *	  @return	void
	 */
	function _pagefoot(&$pdf, $object, $outputlangs, $hidefreetext=0)
	{
		return pdf_pagefoot(
						$pdf, $outputlangs, 'USER_FREE_TEXT', 
						$this->emetteur, $this->marge_basse, $this->marge_gauche, $this->page_hauteur, 
						$object, 0, $hidefreetext
		);
	}
}